import 'package:flutter/material.dart';
import 'package:naim/classes/assets.dart';

enum AlertDialogType {
  SUCCESS,
  ERROR,
  WARNING,
  INFO,
}

class CustomAlertDialog extends StatelessWidget {
  final AlertDialogType type;
  final String title;
  final String content;
  final Widget? icon;
  final String buttonLabel;
  final bool hasCancel;
  final TextStyle titleStyle = TextStyle(
      fontSize: 20.0, color: Colors.black, fontWeight: FontWeight.bold);

  CustomAlertDialog(
      {Key? key,
      this.title = "Successful",
      required this.content,
      this.icon,
      this.type = AlertDialogType.INFO,
      this.buttonLabel = "Ok",
      this.hasCancel = false})
      : super(key: key);

  @override
  Widget build(BuildContext context) {
    return Material(
        type: MaterialType.transparency,
        child: Container(
          alignment: Alignment.center,
          child: Container(
            margin: const EdgeInsets.all(8.0),
            padding: const EdgeInsets.all(20.0),
            decoration: BoxDecoration(
              borderRadius: BorderRadius.circular(20.0),
              color: appColors.backgroundColor(context),
            ),
            child: Column(
              mainAxisSize: MainAxisSize.min,
              children: <Widget>[
                const SizedBox(height: 10.0),
                icon ??
                    Icon(
                      _getIconForType(type),
                      color: _getColorForType(type),
                      size: 50,
                    ),
                const SizedBox(height: 10.0),
                Text(
                  title,
                  style: TextStyle(
                      fontSize: 20.0,
                      color: appColors.armyColor(context),
                      fontWeight: FontWeight.bold),
                  textAlign: TextAlign.center,
                ),
                const Divider(),
                Text(
                  content,
                  style:
                      TextStyle(color: appColors.textColorSecondary(context)),
                  textAlign: TextAlign.center,
                ),
                const SizedBox(height: 40.0),
                SizedBox(
                  width: double.infinity,
                  child: hasCancel
                      ? Row(
                          mainAxisAlignment: MainAxisAlignment.center,
                          children: [
                              getElevatedButton(
                                  appColors.armyColor(context),
                                  buttonLabel,
                                  () => Navigator.pop(context, true)),
                              const SizedBox(width: 20),
                              getOutlinedButton(
                                  appColors.armyColor(context), "Cancel", () {
                                Navigator.pop(context, false);
                              })
                            ])
                      : FlatButton(
                          padding: const EdgeInsets.all(5.0),
                          child: Text(
                            buttonLabel,
                            style:
                                TextStyle(color: appColors.textColor(context)),
                          ),
                          onPressed: () => Navigator.pop(context, true),
                        ),
                ),
              ],
            ),
          ),
        ));
  }

  IconData _getIconForType(AlertDialogType type) {
    switch (type) {
      case AlertDialogType.WARNING:
        return Icons.warning;
      case AlertDialogType.SUCCESS:
        return Icons.check_circle;
      case AlertDialogType.ERROR:
        return Icons.error;
      case AlertDialogType.INFO:
      default:
        return Icons.info_outline;
    }
  }

  Color _getColorForType(AlertDialogType type) {
    switch (type) {
      case AlertDialogType.WARNING:
        return Colors.orange;
      case AlertDialogType.SUCCESS:
        return Colors.green;
      case AlertDialogType.ERROR:
        return Colors.red;
      case AlertDialogType.INFO:
      default:
        return Colors.blue;
    }
  }
}

showAlert(
    BuildContext context, AlertDialogType type, String title, String content) async{
  return await showDialog(
    context: context,
    builder: (BuildContext context) {
      return CustomAlertDialog(
        type: type,
        title: title,
        content: content,
      );
    },
  );
}

showOptionAlert(BuildContext context, AlertDialogType type, String title,
    String content, String confirmLabel) async {
  return await showDialog(
    context: context,
    builder: (BuildContext context) {
      return CustomAlertDialog(
        type: type,
        title: title,
        content: content,
        hasCancel: true,
        buttonLabel: confirmLabel,
      );
    },
  );
}

Widget getOutlinedButton(Color color, String label, VoidCallback func) {
  return OutlinedButton(
    style: OutlinedButton.styleFrom(
        side: BorderSide(
          color: color,
          width: 2,
        ),
        shape:
            RoundedRectangleBorder(borderRadius: BorderRadius.circular(20.0))),
    onPressed: func,
    child: Text(label,
        style: TextStyle(
            fontSize: 14,
            fontWeight: FontWeight.bold,
            letterSpacing: 2.2,
            color: color)),
  );
}

Widget getElevatedButton(Color color, String label, VoidCallback func) {
  return ElevatedButton(
    style: ButtonStyle(
      backgroundColor: MaterialStateProperty.all<Color>(color),
      shape: MaterialStateProperty.all<RoundedRectangleBorder>(
          RoundedRectangleBorder(borderRadius: BorderRadius.circular(20.0))),
    ),
    child: Text(label,
        style: const TextStyle(
            fontSize: 14,
            fontWeight: FontWeight.bold,
            letterSpacing: 2.2,
            color: Colors.white)),
    onPressed: func,
  );
}

Widget getTextButton(Color color, String label, VoidCallback func) {
  return TextButton(
      onPressed: func,
      child: Text(label,
          style: TextStyle(
              fontSize: 14, fontWeight: FontWeight.bold, color: color)));
}
