/*
 * Copyright 2018 Google
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#import <Foundation/Foundation.h>

#import "FirebaseAuth/Sources/Backend/FIRAuthRPCRequest.h"
#import "FirebaseAuth/Sources/Backend/FIRIdentityToolkitRequest.h"

NS_ASSUME_NONNULL_BEGIN

/** @class FIRSignInWithGameCenterRequest
    @brief The request to sign in with Game Center account
 */
@interface FIRSignInWithGameCenterRequest : FIRIdentityToolkitRequest <FIRAuthRPCRequest>

/** @property playerID
    @brief The playerID to verify.
 */
@property(nonatomic, copy) NSString *playerID;

/** @property publicKeyURL
    @brief The URL for the public encryption key.
 */
@property(nonatomic, copy) NSURL *publicKeyURL;

/** @property signature
    @brief The verification signature data generated by Game Center.
 */
@property(nonatomic, copy) NSData *signature;

/** @property salt
    @brief A random strong used to compute the hash and keep it randomized.
 */
@property(nonatomic, copy) NSData *salt;

/** @property timestamp
    @brief The date and time that the signature was created.
 */
@property(nonatomic, assign) uint64_t timestamp;

/** @property accessToken
    @brief The STS Access Token for the authenticated user, only needed for linking the user.
 */
@property(nonatomic, copy, nullable) NSString *accessToken;

/** @property displayName
    @brief The display name of the local Game Center player.
 */
@property(nonatomic, copy, nullable) NSString *displayName;

/** @fn initWithEndpoint:requestConfiguration:
    @brief Please use initWithPlayerID:publicKeyURL:signature:salt:timestamp:requestConfiguration:.
 */
- (nullable instancetype)initWithEndpoint:(NSString *)endpoint
                     requestConfiguration:(FIRAuthRequestConfiguration *)requestConfiguration
    NS_UNAVAILABLE;

/** @fn initWithPlayerID:publicKeyURL:signature:salt:timestamp:displayName:requestConfiguration:
    @brief Designated initializer.
    @param playerID The ID of the Game Center player.
    @param publicKeyURL The URL for the public encryption key.
    @param signature The verification signature generated.
    @param salt A random string used to compute the hash and keep it randomized.
    @param timestamp The date and time that the signature was created.
    @param displayName The display name of the Game Center player.
 */
- (nullable instancetype)initWithPlayerID:(NSString *)playerID
                             publicKeyURL:(NSURL *)publicKeyURL
                                signature:(NSData *)signature
                                     salt:(NSData *)salt
                                timestamp:(uint64_t)timestamp
                              displayName:(NSString *)displayName
                     requestConfiguration:(FIRAuthRequestConfiguration *)requestConfiguration
    NS_DESIGNATED_INITIALIZER;

@end

NS_ASSUME_NONNULL_END
