import 'dart:convert';
import 'dart:io';
import 'dart:typed_data';
import 'dart:ui';

import 'package:dio/dio.dart';
import 'package:dropdown_search/dropdown_search.dart';
import 'package:flutter/material.dart';
import 'package:font_awesome_flutter/font_awesome_flutter.dart';
import 'package:geolocator/geolocator.dart';
import 'package:path/path.dart';
import 'package:device_info/device_info.dart';

import 'classes/state_model.dart';
import 'widgets/bezier_widget.dart';
import 'widgets/image_picker_handler.dart';
import 'widgets/input.dart';

class CrimeReportPage extends StatefulWidget {
  @override
  _CrimeReportPageState createState() => _CrimeReportPageState();
}

class _CrimeReportPageState extends State<CrimeReportPage>
    with TickerProviderStateMixin, ImagePickerListener {
  bool _checkboxValue = false;

  final double height = window.physicalSize.height;
  StateModel? selectedState;
  StateModel? selectedCrime;
  List stateData = [];
  List crimeData = [];

  List<int> finalImage = [];
  final scaffoldKey = GlobalKey<ScaffoldState>();
  final formKey = GlobalKey<FormState>();
  ImagePickerHandler? imagePicker;
  File? _image;
  bool imageSet = false;
  bool isUpdating = false;
  AnimationController? _controller;
  final GeolocatorPlatform _geolocatorPlatform = GeolocatorPlatform.instance;
  Widget? selImage = Image.asset(
    "assets/img/blank.jpg",
    fit: BoxFit.cover,
  );
  var screenNameCtrl = TextEditingController();
  var phoneCtrl = TextEditingController();
  var reportCtrl = TextEditingController();
  var loadingState = false;
  var loadingCrimes = false;

  @override
  void initState() {
    // TODO: implement initState
    loadStates();
    loadCrimeList();
    super.initState();
    _controller = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 600),
    );
    imagePicker = ImagePickerHandler(this, _controller);
    imagePicker!.init(false);
  }

  @override
  void dispose() {
    _controller!.dispose();

    super.dispose();
  }

  @override
  userImage(File image) {
    setState(() {
      _image = image;
      imageSet = true;
      selImage = Image.file(
        image,
        fit: BoxFit.cover,
      );
    });
    compressFile(image).then((value) {
      finalImage = value;
    });
  }

  void setPhoto(BuildContext context) async {
    imagePicker!.showDialog(context);
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
        key: scaffoldKey,
        appBar: AppBar(
          elevation: 0,
          backgroundColor: Colors.transparent,
          leading: IconButton(
              onPressed: () {
                Navigator.of(context).pop();
              },
              icon: const Icon(FontAwesomeIcons.caretLeft)),
        ),
        extendBodyBehindAppBar: true,
        body: Stack(
          children: [
            BezierWidget(),
            SingleChildScrollView(
              child: SafeArea(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.stretch,
                  children: [
                    const SizedBox(
                      height: 50,
                    ),
                    Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Padding(
                          padding: const EdgeInsets.all(8.0),
                          child: Input(
                              placeholder: "Screen Name...",
                              prefixIcon: const Icon(Icons.school, size: 20),
                              controller: screenNameCtrl),
                        ),
                        Padding(
                          padding: const EdgeInsets.all(8.0),
                          child: Input(
                            placeholder: "Phone (Optional)...",
                            prefixIcon: Icon(Icons.phone, size: 20),
                            controller: phoneCtrl,
                            inputType: TextInputType.phone,
                          ),
                        ),
                        Padding(
                          padding: const EdgeInsets.all(8.0),
                          child: loadingState
                              ? const Text("Loading States...",
                                  style: TextStyle(
                                      fontStyle: FontStyle.italic,
                                      fontSize: 13))
                              : DropdownSearch<StateModel>(
                                  showSelectedItems: true,
                                  compareFn: (i, s) => i?.isEqual(s) ?? false,
                                  label: "Select State",
                                  onFind: (String? filter) =>
                                      getStateDataFrom(filter),
                                  onChanged: (data) {
                                    //print(data);
                                    setState(() {
                                      selectedState = data;
                                    });
                                  },
                                  showSearchBox: true,
                                  dropdownBuilder: _customDropDownStateModel,
                                  popupItemBuilder:
                                      _customPopupItemBuilderStateModel,
                                ),
                        ),
                        Padding(
                          padding: const EdgeInsets.all(8.0),
                          child: loadingCrimes
                              ? const Text("Loading Crimes...",
                                  style: TextStyle(
                                      fontStyle: FontStyle.italic,
                                      fontSize: 13))
                              : DropdownSearch<StateModel>(
                                  showSelectedItems: true,
                                  compareFn: (i, s) => i?.isEqual(s) ?? false,
                                  label: "Select Crime Type",
                                  onFind: (String? filter) =>
                                      getCrimeDataFrom(filter),
                                  onChanged: (data) {
                                    //print(data);
                                    setState(() {
                                      selectedCrime = data;
                                    });
                                  },
                                  showSearchBox: true,
                                  dropdownBuilder: _customDropDownCrimeModel,
                                  popupItemBuilder:
                                      _customPopupItemBuilderCrimeModel,
                                ),
                        ),
                        Padding(
                            padding: const EdgeInsets.all(8.0),
                            child: Input(
                                placeholder: "Enter Message...",
                                prefixIcon: const Icon(Icons.mail, size: 20),
                                inputType: TextInputType.multiline,
                                controller: reportCtrl)),
                        Padding(
                          padding: const EdgeInsets.all(8.0),
                          child: Container(
                            height: 150,
                            child: Stack(
                              children: [
                                Container(
                                  height: double.infinity,
                                  width: double.infinity,
                                  child: selImage,
                                ),
                                Center(
                                  child: Row(
                                    mainAxisAlignment: MainAxisAlignment.center,
                                    crossAxisAlignment:
                                        CrossAxisAlignment.center,
                                    children: [
                                      MaterialButton(
                                        minWidth: 0,
                                        elevation: 0.5,
                                        color: Colors.white,
                                        child: const Icon(
                                          Icons.camera_alt_outlined,
                                          color: Colors.green,
                                        ),
                                        shape: RoundedRectangleBorder(
                                          borderRadius:
                                              BorderRadius.circular(20.0),
                                        ),
                                        onPressed: () {
                                          setPhoto(context);
                                        },
                                      ),
                                      imageSet
                                          ? MaterialButton(
                                              minWidth: 0,
                                              elevation: 0.5,
                                              color: Colors.white,
                                              child: const Icon(
                                                Icons.remove,
                                                color: Colors.red,
                                              ),
                                              shape: RoundedRectangleBorder(
                                                borderRadius:
                                                    BorderRadius.circular(20.0),
                                              ),
                                              onPressed: () {
                                                setState(() {
                                                  imageSet = false;
                                                  selImage = Image.asset(
                                                    "assets/img/blank.jpg",
                                                    fit: BoxFit.cover,
                                                  );
                                                });
                                              },
                                            )
                                          : Container()
                                    ],
                                  ),
                                ),
                              ],
                            ),
                          ),
                        ),
                      ],
                    ),
                    Center(
                      child: ElevatedButton(
                        style: primaryButtonStyle,
                        onPressed: () {
                          // Respond to button press
                          if (!isUpdating) {
                            setState(() {
                              isUpdating = true;
                            });
                            print("attempting to submit report");

                            submitCrimeReport();
                            print("call the function to submit report");
                          }
                        },
                        child: Padding(
                            padding: const EdgeInsets.only(
                                left: 32.0, right: 32.0, top: 12, bottom: 12),
                            child: Text(
                                isUpdating ? "Please wait..." : "Submit Report",
                                style: const TextStyle(fontSize: 14.0))),
                      ),
                    ),
                  ],
                ),
              ),
            ),
          ],
        ));
  }

  Widget _customDropDownStateModel(BuildContext context, StateModel? item) {
    if (item == null) {
      return Container();
    }

    return Container(
      child: ListTile(
          contentPadding: const EdgeInsets.all(0),
          leading:
              Icon(FontAwesomeIcons.mapPin, size: 20, color: Colors.blueGrey),
          title: Text(item.name)),
    );
  }

  Widget _customDropDownCrimeModel(BuildContext context, StateModel? item) {
    if (item == null) {
      return Container();
    }

    return Container(
      child: ListTile(
          contentPadding: const EdgeInsets.all(0),
          leading: const Icon(FontAwesomeIcons.bell,
              size: 20, color: Colors.blueGrey),
          title: Text(item.name)),
    );
  }

  Widget _customPopupItemBuilderStateModel(
      BuildContext context, StateModel? item, bool isSelected) {
    return Container(
      margin: const EdgeInsets.symmetric(horizontal: 8),
      decoration: !isSelected
          ? null
          : BoxDecoration(
              border: Border.all(color: Colors.blueGrey),
              borderRadius: BorderRadius.circular(15),
              color: Colors.white,
            ),
      child: ListTile(
        selected: isSelected,
        title: Text(item?.name ?? ''),
        leading:
            Icon(FontAwesomeIcons.mapPin, size: 20, color: Colors.blueGrey),
      ),
    );
  }

  Widget _customPopupItemBuilderCrimeModel(
      BuildContext context, StateModel? item, bool isSelected) {
    return Container(
      margin: const EdgeInsets.symmetric(horizontal: 8),
      decoration: !isSelected
          ? null
          : BoxDecoration(
              border: Border.all(color: Colors.blueGrey),
              borderRadius: BorderRadius.circular(15),
              color: Colors.white,
            ),
      child: ListTile(
        selected: isSelected,
        title: Text(item?.name ?? ''),
        leading: Icon(FontAwesomeIcons.bell, size: 20, color: Colors.blueGrey),
      ),
    );
  }

  loadStates() async {
    setState(() {
      loadingState = true;
    });
    var val = [];
    try {
      var dio = new Dio();
      dio.options.baseUrl = appApi;

      var formData = {};
      //print(dio.options.baseUrl+ "crimereport/getallstates");
      var response = await dio.post(
        "crimereport/getallstates",
        data: json.encode(formData),
        options: Options(
          method: 'POST',
          responseType: ResponseType.plain,
          headers: {
            HttpHeaders.contentLengthHeader:
                formData.length, // set content-length
          },
        ),
      );
      //print("Response status: ${response.statusCode}");
      //print("Response data: ${response.data}");
      var data = json.decode(response.data);

      if (data["result"] != null) {
        if (data["result"] == "success") {
          var dat = [];
          data['data'].forEach((k) {
            dat.add({'id': k["state"], "name": k['state']});
          });
          //print(dat);
          setState(() {
            stateData = dat;
            loadingState = false;
          });
        }
      }
    } catch (e) {
      print(e.toString());
      setState(() {
        loadingState = false;
      });
    }
    return val;
  }

  loadCrimeList() async {
    setState(() {
      loadingCrimes = true;
    });
    var val = [];
    try {
      var dio = new Dio();
      dio.options.baseUrl = appApi;

      var formData = {};
      //print(dio.options.baseUrl+ "crimereport/getcrimelist");
      var response = await dio.post(
        "crimereport/getcrimelist",
        data: json.encode(formData),
        options: Options(
          method: 'POST',
          responseType: ResponseType.plain,
          headers: {
            HttpHeaders.contentLengthHeader:
                formData.length, // set content-length
          },
        ),
      );
      //print("Response status: ${response.statusCode}");
      //print("Response data: ${response.data}");
      var data = json.decode(response.data);

      if (data["result"] != null) {
        if (data["result"] == "success") {
          var dat = [];
          data['data'].forEach((k) {
            dat.add({'id': k["name"], "name": k['name']});
          });
          //print(dat);
          setState(() {
            crimeData = dat;
            loadingCrimes = false;
          });
        }
      }
    } catch (e) {
      print(e.toString());
      setState(() {
        loadingCrimes = false;
      });
    }
    return val;
  }

  submitCrimeReport() async {
    var file = "";
    var file_name = "";
    if (imageSet) {
      String base64Image = base64Encode(finalImage);
      file = "data:image/png;base64," + base64Image;
      file_name = basename(_image!.path);
    }

    var pos = await getCurrentPosition();
    print("got postions.....");
    if (pos.altitude == 0 &&
        pos.speed == 0 &&
        pos.latitude == 0 &&
        pos.longitude == 0) {
      setState(() {
        isUpdating = false;
      });
      print("has no position sorry.....");
      return;
    }
    if (screenNameCtrl.text == "") {
      showAlert(this.context, AlertDialogType.ERROR, "",
          "Screen name cannot be empty");
      setState(() {
        isUpdating = false;
      });
      print("empty screen name sorry.....");
      return;
    }
    if (reportCtrl.text == "") {
      showAlert(this.context, AlertDialogType.ERROR, "",
          "Your message cannot be empty");
      setState(() {
        isUpdating = false;
      });
      print("empty report sorry.....");
      return;
    }
    if (selectedCrime == null) {
      showAlert(this.context, AlertDialogType.ERROR, "",
          "Please select a crime type");
      setState(() {
        isUpdating = false;
      });
      print("empty Crime name sorry.....");
      return;
    }
    if (selectedState == null) {
      showAlert(
          this.context, AlertDialogType.ERROR, "", "Please select a State");
      setState(() {
        isUpdating = false;
      });
      print("empty state name sorry.....");
      return;
    }
    try {
      var dio = new Dio();
      dio.options.baseUrl = appApi;
      print("attemping post.....");

      String device = "";
      DeviceInfoPlugin deviceInfo = DeviceInfoPlugin();
      if (Platform.isIOS) {
        IosDeviceInfo iosInfo = await deviceInfo.iosInfo;
        device = iosInfo.utsname.machine;
      } else {
        AndroidDeviceInfo androidInfo = await deviceInfo.androidInfo;
        device = androidInfo.model;
      }

      var formData = {
        "sname": screenNameCtrl.text,
        "sphone": phoneCtrl.text,
        "email": "",
        "state": selectedState!.name,
        "file": file,
        "file_name": file_name,
        "address": "",
        "crimeType": selectedCrime!.name,
        "report": reportCtrl.text,
        "device": Platform.isIOS ? "ios" : "android",
        "os": Platform.isIOS ? "ios" : "android",
        "browser": device,
        "longitude": pos.longitude,
        "latitude": pos.latitude
      };
      print(dio.options.baseUrl + "crimereport/createcrimerport");
      var response = await dio.post(
        "crimereport/createcrimerport",
        data: json.encode(formData),
        options: Options(
          method: 'POST',
          responseType: ResponseType.plain,
          headers: {
            HttpHeaders.contentLengthHeader:
                formData.length, // set content-length
          },
        ),
      );
      print("Response status: ${response.statusCode}");
      print("Response data: ${response.data}");
      var data = json.decode(response.data);
      print(data["result"]);

      if (data["result"] != null) {
        print("it got the success not null");
        if (data["result"] == "success") {
          print("finally");

          showDialog(
            context: this.context,
            builder: (BuildContext context) {
              return Material(
                  type: MaterialType.transparency,
                  child: Container(
                    alignment: Alignment.center,
                    child: Container(
                      margin: const EdgeInsets.all(8.0),
                      padding: const EdgeInsets.all(20.0),
                      decoration: BoxDecoration(
                        borderRadius: BorderRadius.circular(20.0),
                        color: Colors.white,
                      ),
                      child: Column(
                        mainAxisSize: MainAxisSize.min,
                        children: <Widget>[
                          const SizedBox(height: 10.0),
                          const Icon(
                            Icons.check_circle,
                            color: Colors.green,
                            size: 50,
                          ),
                          const SizedBox(height: 10.0),
                          const Text(
                            "",
                            style: TextStyle(
                                fontSize: 20.0,
                                color: Colors.black,
                                fontWeight: FontWeight.bold),
                            textAlign: TextAlign.center,
                          ),
                          Divider(),
                          Text(
                            "Your report has been submitted",
                            textAlign: TextAlign.center,
                          ),
                          SizedBox(height: 40.0),
                          SizedBox(
                            width: double.infinity,
                            child: FlatButton(
                              padding: const EdgeInsets.all(5.0),
                              child: const Text("OK"),
                              onPressed: () {
                                Navigator.pop(context, true);
                                Navigator.pop(this.context, true);
                              },
                            ),
                          ),
                        ],
                      ),
                    ),
                  ));
            },
          );
          //Navigator.of(this.context).pop();
        }
      }
      print('oh shit');
    } on DioError catch (e) {
      print(e.toString());
      print(e.message);
      print(e.response);
      //Navigator.push(this.context,MaterialPageRoute(builder: (context) => Htmlview(e.response!.data.toString())));
      showAlert(
          this.context, AlertDialogType.ERROR, "Oops", "Something went wrong");
      setState(() {
        isUpdating = false;
      });
    }
  }

  Future<List<StateModel>> getStateDataFrom(String? filter) async {
    var bm = StateModel.fromJsonList(stateData);
    if (filter != "") {
      bm = bm.where((i) => i.name.contains(filter!)).toList();
    }
    return StateModel.fromJsonList(stateData);
  }

  Future<List<StateModel>> getCrimeDataFrom(String? filter) async {
    var bm = StateModel.fromJsonList(crimeData);
    if (filter != "") {
      bm = bm.where((i) => i.name.contains(filter!)).toList();
    }
    return StateModel.fromJsonList(crimeData);
  }

  Future<Position> getCurrentPosition() async {
    final hasPermission = await _handlePermission();

    if (!hasPermission) {
      Position p = Position(
          longitude: 0,
          latitude: 0,
          timestamp: DateTime.now(),
          accuracy: 0,
          altitude: 0,
          heading: 0,
          speed: 0,
          speedAccuracy: 0);
      return p;
    }
    DeviceInfoPlugin deviceInfo = DeviceInfoPlugin();
    var isPhysical = true;
    if (Platform.isIOS) {
      IosDeviceInfo iosInfo = await deviceInfo.iosInfo;
      isPhysical = iosInfo.isPhysicalDevice;
    } else {
      AndroidDeviceInfo androidInfo = await deviceInfo.androidInfo;
      isPhysical = androidInfo.isPhysicalDevice;
    }
    if (isPhysical) {
      final position = await _geolocatorPlatform.getCurrentPosition();
      return position;
    } else {
      var p = Position(
          longitude: 3.532,
          latitude: 4.342,
          timestamp: DateTime.now(),
          accuracy: 1,
          altitude: 1,
          heading: 0,
          speed: 0,
          speedAccuracy: 0);
      return p;
    }
  }

  showAlert(BuildContext context, AlertDialogType type, String title,
      String content) {
    showDialog(
      context: context,
      builder: (BuildContext context) {
        return CustomAlertDialog(
          type: type,
          title: title,
          content: content,
        );
      },
    );
  }

  Future<bool> _handlePermission() async {
    bool serviceEnabled;
    LocationPermission permission;

    // Test if location services are enabled.
    serviceEnabled = await _geolocatorPlatform.isLocationServiceEnabled();
    if (!serviceEnabled) {
      // Location services are not enabled don't continue
      // accessing the position and request users of the
      // App to enable the location services.
      showAlert(this.context, AlertDialogType.INFO, "Service Disabled",
          "Location Services are disabled. Please enable to continue");
      print("return  Service Disabled...");
      return false;
    }

    permission = await _geolocatorPlatform.checkPermission();

    if (permission == LocationPermission.denied) {
      permission = await _geolocatorPlatform.requestPermission();
      if (permission == LocationPermission.denied) {
        // Permissions are denied, next time you could try
        // requesting permissions again (this is also where
        // Android's shouldShowRequestPermissionRationale
        // returned true. According to Android guidelines
        // your App should show an explanatory UI now.
        showAlert(this.context, AlertDialogType.INFO, "Permission Denied",
            "Permission to Location Services where denied. Please allow to continue");
        print("return Permission Denied...");
        return false;
      }
    }

    if (permission == LocationPermission.deniedForever) {
      // Permissions are denied forever, handle appropriately.
      showAlert(this.context, AlertDialogType.INFO, "Permission Denied",
          "Permission to Location Services have been denied for this app. Please allow to continue");

      return false;
    }

    return true;
  }
}
